#!/usr/bin/env bash

okPrint() {
	echo -e "\e[32m${1}\e[0m"
}

warnPrint() {
	echo -e "\e[33m${1}\e[0m"
}

failPrint() {
	echo -e "\e[91m${1}\e[0m"
	exit 1
}

fileExists() {
	if [ -s "$1" ]; then
		return 0 # true
	else
		return 1 # false
	fi
}

daemonRunning() {
    pidof letsencrypt.live.cgi > /dev/null
    return $?
}

checkRoot() {
	if (( EUID != 0 )); then
		failPrint "Please run installer as root"
	fi
	okPrint "Running installer as root"
}

checkOsVersion() {
	if ! fileExists "/etc/redhat-release"; then
        failPrint "/etc/redhat-release missing"
	fi
	if grep -q "CentOS release 5." /etc/redhat-release || grep -q "CloudLinux Server release 5" /etc/redhat-release; then
	    failPrint "Requires CentOS/CloudLinux 6.0 minimum to install"
	fi
	okPrint "OS version OK"
}

checkCpanelVersion() {
    if ! fileExists "/usr/local/cpanel/cpanel"; then
		failPrint "Cannot find cpanel installed"
    fi
}

checkLicence() {
    if ! fileExists "/etc/letsencrypt-cpanel.licence"; then
        warnPrint "No licence file detected at /etc/letsencrypt-cpanel.licence"
        warnPrint "Fetching new trial licence ... "

        local trial
        trial="$(curl -sS --data hostname="${HOSTNAME}" https://cpanel.fleetssl.com/generatetrial)"
        if [ ! "${trial:0:1}" == "{" ]; then
            warnPrint "Error downloading trial licence: ${trial}"
            failPrint "Please install a licence or contact support"
        fi

        echo "${trial}" > /etc/letsencrypt-cpanel.licence
        chmod 0400 /etc/letsencrypt-cpanel.licence
    fi

    if ! fileExists "/etc/letsencrypt-cpanel.licence"; then
        failPrint "No licence file present"
    fi

    okPrint "Licence file present"
}

stopService() {
    service letsencrypt-cpanel stop > /dev/null
    if daemonRunning; then
        warnPrint "Daemon service not stopped successfully, attempting manual shutdown"
        killall -TERM letsencrypt.live.cgi
        sleep 5s
        if daemonRunning; then
            warnPrint "Manual shutdown not successful, killing process"
            killall -9 letsencrypt.live.cgi
        fi
    fi
    okPrint "FleetSSL cPanel service daemon stopped"
}

okPrint "*** By running this installer, you indicate that you have read the end-user\n licence agreement (https://cpanel.fleetssl.com/eula) and agree to all of its terms, as stated. ***\n"

checkRoot
checkOsVersion
checkCpanelVersion
checkLicence

stopService
